
=head1 NAME

SuikaWiki::DB::Util::Error - SuikaWiki WikiDatabase Implementation : Common Error Definitions

=head1 DESCRIPTION

This module defines common WikiDatabase errors, inheriting
Message::Util::Error.

This module is part of SuikaWiki.

=cut

package SuikaWiki::DB::Util::Error;
use strict;
our $VERSION=do{my @r=(q$Revision: 1.10 $=~/\d+/g);sprintf "%d."."%02d" x $#r,@r};
require Message::Util::Error;
our @ISA = 'Message::Util::Error';

sub ___error_def () {+{
    ## Error levels:
    ## - stop : Cannot continue the operation (but other operations 
    ##          might be able to be performed)
    ## - fatal: Cannot continue any operation anymore
    ## - warn : Simple warning or harmless error
    ## - log  : Useful event message for logging
    ## - detaillog: Detailed logging message
    ## - debug: Debug message
    DB_METHOD_NOT_IMPLEMENTED => {
      level => 'stop',
      description => q(%t (name => method, suffix => {: });Method not implemented),
    },
    DB_METHOD_PARAMETER_REQUIRED => {
      level => 'fatal',
      description => q(%t (name => method, suffix => {: });Required parameter "%t (name => parameter);" not specified),
    },
    DB_OPEN => {
      level => 'fatal',
      description => q(Can't open database%t(name=>msg,prefix=>{: });),
    },
    DB_UNSUPPORTED_FORMAT => {
      level => 'stop',
      description => q(Unsupported format of database),
    },
    DB_SAVE => {
      level => 'stop',
      description => q(Can't save database, so data might be lost: %t (name => file);: %t (name => msg);),
    },
    DB_WROTE => {
      level => 'detaillog',
      description => '%t (name => prop, suffix => {: });Wrote successfully',     
    },
    DB_CLOSE => {
      level => 'fatal',
      description => q(Can't close database%t(name=>msg,prefix=>{: });),
    },
    DB_CLOSED => {
      level => 'detaillog',
      description => q(Database is closed successfully),
    }, 
    DB_DESTROY => {
      level => 'detaillog',
      description => q(Instance of the database is destroyed),
    },
    DIR_MAKE_FAILURE => {
      level => 'stop',
      description => q(%t (name => dir, suffix => {: });Can't create directory%t (name => msg, prefix => {: });),
    },
    FILE_READ_FAILURE => {
      level => 'stop',
      description => q(%t (name => file, suffix => {: });Can't open file to read%t (name => msg, prefix => {: });),
    },
    FILE_REMOVE_FAILURE => {
      level => 'stop',
      description => q(%t (name => file, suffix => {: });Can't remove file%t(name => msg, prefix => {: });),
    },
    FILE_WRITE_FAILURE => {
      level => 'stop',
      description => q(%t (name => file, suffix => {: });Can't open file to write%t (name => msg, prefix => {: });),
    },
    FILE_WRITE_FAILURE__NO_DIR => {
      level => 'stop',
      description => q(%t (name => file, suffix => {: });Can't write file on the directory: directory not found),
    },
    INFO_DB_PROP_OPENED => {
      level => 'detaillog',
      description => q(%t (name => method);: Sub-database "%t (name => prop);" opened),
    },
    INFO_DB_PROP_CLOSED => {
      level => 'detaillog',
      description => q(%t (name => method);: Sub-database "%t (name => prop);" closed),
    },
    INFO_DB_PROP_NOT_MODIFIED => {
      level => 'detaillog',
      description => q(%t (name => method);: Sub-database "%t (name => prop);" not modified),
    },
    INFO_DB_PROP_WROTE => {
      level => 'detaillog',
      description => q(%t (name => method);: Sub-database "%t (name => prop);" saved),
    },
    STOP_DB_PROP_CANT_OPEN => {
      level => 'stop',
      description => q(%t (name => method);: Sub-database "%t (name => prop);" cannot be opened%t (prefix => {: }, name => file);%t (prefix => {: }, name => msg);),
    },
    STOP_DB_PROP_CANT_OPEN__NOT_DEFINED => {
      level => 'stop',
      description => q(%t (name => method);: Sub-database "%t (name => prop);" cannot be opened: sub-database not defined),
    },
    STOP_DB_PROP_CANT_OPEN__REQUIRED_METHOD_NOT_DEFINED => {
      level => 'stop',
      description => q(%t (name => method);: Sub-database "%t (name => prop);" cannot be opened: required method "%t (name => method-name);" not defined),
    },
    STOP_DB_PROP_CANT_SAVE => {
      level => 'stop',
      description => q(%t (name => method);: Sub-database "%t (name => prop);" cannot be saved%t (prefix => {: }, name => file);%t (prefix => {: }, name => msg);),
    },
    STOP_SET_INVALID_VALUE => {
      level => 'stop',
      description => q(%t (name => method);: Given value is not valid for this database),
    },
    LOCK_START => {
      level => 'fatal',
      description => q(Can't lock the database%t(name=>msg,prefix=>{: });),
      http_status => 423,
    },
    LOCK_END => {
      level => 'warn',
      description => q(Can't unlock the database%t(name=>msg,prefix=>{: });),
    },
    KEY_INVALID_NAME => {
      level => 'stop',
      description => q(%key;: Invalid key name),
    },
    KEY_INVALID_NS_NAME => {
      level => 'stop',
      description => q(%key;: Invalid key namespace),
    },                                
    KEY_SAVE => {
      level => 'stop',
      description => q(Can't save to the database%t(name=>msg,prefix=>{: });),
    },
    KEY_SAVE_LOCKED => {
      level => 'stop',
      description => q(Can't save to the database: It has been locked by third party),
      http_status => 423,
    },         
    UNKNOWN => {
      level => 'fatal',
      description => 'Unknown error: %name;%t(name=>msg,prefix=>{: });',
    },
    ERROR_REPORTED => {
      level => 'fatal', # dummy error
      description => 'Fatal error occured has been reported',
    },
}}

sub _FORMATTER_PACKAGE_ () { 'SuikaWiki::DB::Util::Error::formatter' }

package SuikaWiki::DB::Util::Error::formatter;
push our @ISA, 'Message::Util::Error::formatter';

sub ___rule_def () {+{
  key => {
    after => sub {
      my ($f, $name, $p, $o) = @_;
      if (ref $o->{key}) {
        $p->{-result} .= join '/', map {my $s = $_; $s =~ s!([/\\])!\\$1!; $s}
                                       @{$o->{key}||[]};
      } else {
        $p->{-result} .= defined $o->{key} ?
                           length $o->{key} ? 
                             '<"' . $o->{key} . '">' :
                             defined $p->{empty} ? $p->{empty} : '<empty>':
                           defined $p->{undef} ? $p->{undef} : '<undef>';
      }
    },
  },
}}

=head1 EXAMPLE

  use SuikaWiki::DB::Util::Error;
  try {
    $db->... (...);
  } catch SuikaWiki::DB::Util::Error with {
    my $err = shift;
    warn $err->stringify;
    print LOG $err->stringify;
  };

=head1 SEE ALSO

L<Message::Util::Error>, L<Error>

=head1 LICENSE

Copyright Wakaba <w@suika.fam.cx> 2003.

This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=cut

1; # $Date: 2004/11/08 09:57:50 $
