#!/usr/bin/perl
use strict;
use Getopt::Long;
use lib qw<lib ../lib>;
use BerkeleyDB;
our $VERSION = do{my @r=(q$Revision: 1.3 $=~/\d+/g);sprintf "%d."."%02d" x $#r,@r};

my %opt = (
  db => q<wikidata>,
);
GetOptions (
  'database-dir=s' => \$opt{db},
  'input-file=s' => \$opt{src},
  'output-dir=s' => \$opt{output},
);

$opt{output} ||= $opt{db} . q</page/>;
$opt{src} ||= $opt{db} . q</page/mt--52656665726572.dat>;

print STDERR "Reading $opt{src}...\n";
open my $src, '<', $opt{src} or die "$0: $opt{src}: $!";
local $/ = undef;

print STDERR "Output: $opt{output}\n";
        require SuikaWiki::DB::FileSystem::Count;
my $out = SuikaWiki::DB::FileSystem::Count->new 
          (base_directory => $opt{output},
           directory_suffix => '.ns',
           file_suffix    => '.ref',
           auto_mkdir => 1);

my $source = scalar <$src>;
$source =~ s!^\#\?SuikaWikiMetaInfo/0.9[^\x02]*\x02!!;

my $i = 0;
for my $old (map {[split /\x1F/, $_, 2]} split /\x1E/, $source) {
  my $page = [split m#//#, $old->[0]];
  my $referer = $out->get ('referer', $page);
  my %newref = split /"/, $old->[1];
  for (grep {$_} keys %newref) {
    $referer->{$_} += $newref{$_} if $_ and $newref{$_};
  }
  $out->set ('referer', $page => $referer);
  
  print STDERR '*' unless ++$i % 10;
  print STDERR "\n" unless $i % 500;
}

close $src;
$out->close;

print STDERR "\n";

=head1 NAME

referer2to3.pl - SuikaWiki: Converting SuikaWiki 2 Referer Database into SuikaWiki 3 Referer Database

=head1 SYNOPSIS

  referer2to3.pl [--database-dir=<dir>] [--input-file=<file>] [--output-dir=<dir>]

=head1 DESCRIPTION

C<referer2to3.pl> converts SuikaWiki 2 referer database into SuikaWiki 3
database.  SuikaWiki 2 stores received HTTP Referer information
into SuikaWikiMetaInfo/0.9 database in wiki database directory.
Referer module of SuikaWiki 3 stores them, in default configuration,
into BerkeleyDB with different sub-format, so convertion required
to upgrade to SuikaWiki 3.

This script is part of SuikaWiki.

=head1 OPTIONS

=over 4

=item --database-dir=I<database-dir> (Default: C<./wikidata>)

Wiki database directory.  This option only used when either
C<--input-file> or C<--output-file> option is not specified.

=item --input-file=I<file> (Default: C<I<database-dir>/page/mt--52656665726572.dat>)

Source (SuikaWiki 2) referer database file.

=item --output-dir=I<dir> (Default: C<I<database-dir>/page/>)

New (SuikaWiki 3 BerkeleyDB) referer database directory.

=back

=head1 SEE ALSO

C<SuikaWiki::Plugin>, SuikaWiki:WikiPlugin
<http://suika.fam.cx/~wakaba/-temp/wiki/wiki?WikiPlugin>.

=head1 LICENSE

Copyright 2004 Wakaba <w@suika.fam.cx>.  All rights reserved.

This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=cut

1; # $Date: 2004/04/17 04:15:17 $
